﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace K01DASample
{
    public partial class DevForm : Form
    {
        private short DevId;               //Device Id

        private byte[] IOStatus;           //DIO Status
        
        public DevForm()
        {
            InitializeComponent();

            IOStatus = new byte[2];
        }

        public void SetId(short Id)
        {
            short retErr;

            DevId = Id;

            this.Text = "ID = " + DevId.ToString() + "  Panel";
            retErr = TUSBKDA.TUSBK01DA_DIO_Status(DevId, IOStatus);
            if (retErr != 0)
            {
                MessageBox.Show(TUSBKDA.GetErrMessage(retErr));
                this.Hide();
                return;
            }
            else
            {
                IOSEL1.SelectedIndex = IOStatus[0];
                IOSEL2.SelectedIndex = IOStatus[1];
            }

            ExtClock.Text = "1000";
            ExtClockValueCheck();
            IntClockValueCheck();
            SelIntClk.Checked = true;
            Timer1.Interval = 100;
            Timer1.Enabled = true;
            WAVE_SHAPE.SelectedIndex = 0;
        }

        private void DevForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            Timer1.Enabled = false;
        }

        private void DevForm_VisibleChanged(object sender, EventArgs e)
        {
            if( this.Visible == false)
                Timer1.Enabled = false;
        }

        // Enable Disable change of controls
        private void EnableDisableCont(Boolean StartStop)
        {
            if (StartStop) //Start ( on data recording )
            {
                DacStop.Enabled = true;
                DacStart.Enabled = false;
                SelExtClock.Enabled = false;
                SelIntClk.Enabled = false;
                WAVE_SHAPE.Enabled = false;
                AMPL.Enabled = false;
                OFFSET.Enabled = false;
                DATA_LENG.Enabled = false;
            }
            else
            {
                DacStop.Enabled = false;
                DacStart.Enabled = true;
                SelExtClock.Enabled = true;
                SelIntClk.Enabled = true;
                WAVE_SHAPE.Enabled = true;
                AMPL.Enabled = true;
                OFFSET.Enabled = true;
                DATA_LENG.Enabled = true;
            }
        }

        //Digital Port Direction Change
        private void SetDir(byte b)
        {
            byte Direction;
            short retErr;

            if (IOStatus[b] != 0)
                Direction = 1;
            else
                Direction = 0;

            retErr = TUSBKDA.TUSBK01DA_DIO_Dir(DevId, b, Direction);

            if (retErr != 0)
                MessageBox.Show(TUSBKDA.GetErrMessage(retErr));
        }

        //
        private void SetDio(byte b)
        {
            byte data;
            short retErr;

            if (IOStatus[b] == 2)
                data = 1;
            else
                data = 0;

            retErr = TUSBKDA.TUSBK01DA_DIO_Out(DevId, b, data);
            if( retErr != 0 )
                MessageBox.Show(TUSBKDA.GetErrMessage(retErr));
        }

        //setup of Digital IO 1 state
        private void IOSEL1_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch (IOSEL1.SelectedIndex)
            {
                case 0://input
                    IOStatus[0] = (byte) IOSEL1.SelectedIndex;
                    SetDir(0);
                    break;
                case 1://output low
                    IOStatus[0] = (byte) IOSEL1.SelectedIndex;
                    SetDio(0);
                    SetDir(0);
                    break;
                case 2://output high
                    IOStatus[0] = (byte) IOSEL1.SelectedIndex;
                    SetDio(0);
                    SetDir(0);
                    break;
            }
        }

        //setup of Digital IO 2 state
        private void IOSEL2_SelectedIndexChanged(object sender, EventArgs e)
        {
            short retErr;

            switch (IOSEL2.SelectedIndex)
            {
                case 0: //input
                    IOStatus[1] = (byte) IOSEL2.SelectedIndex;
                    SetDir(1);
                    break;
                case 1://output low
                    IOStatus[1] = (byte) IOSEL2.SelectedIndex;
                    SetDio(1);
                    SetDir(1);
                    break;
                case 2://output high
                    IOStatus[1] = (byte)IOSEL2.SelectedIndex;
                    SetDio(1);
                    SetDir(1);
                    break;
                case 3://clock
                    retErr = TUSBKDA.TUSBK01DA_DIO_Status(DevId, IOStatus);
                    if (retErr != 0)
                    {
                        MessageBox.Show(TUSBKDA.GetErrMessage(retErr));
                        this.Hide();
                        return;
                    }
                    else if (IOStatus[1] != 3)
                        IOSEL2.SelectedIndex = IOStatus[1];
                    break;
            }
        }

        private void DIO_Check()
        {
            byte data;
            short retErr;

            retErr = TUSBKDA.TUSBK01DA_DIO_In(DevId, out data);
            if (retErr != 0)
            {
                MessageBox.Show(TUSBKDA.GetErrMessage(retErr));
                this.Hide();
                return;
            }
            D0State.Text = ((data & 0x1) != 0 )? "High" : "Low";
            D1State.Text = ((data & 0x2) != 0) ? "High" : "Low";
        }

        // Ext Clock Out Command
        private void ClockOut_Click(object sender, EventArgs e)
        {
            int Clock;
            short retErr;

            if (int.TryParse(ExtClock.Text, out Clock))
            {
                if (Clock < 1000 || Clock > 500000000)
                    MessageBox.Show("Cycle is outside the range.");
                else
                {
                    retErr = TUSBKDA.TUSBK01DA_ExtTimer_Set(DevId, Clock);
                    if (retErr != 0)
                        MessageBox.Show(TUSBKDA.GetErrMessage(retErr));
                    else
                        IOSEL2.SelectedIndex = 3;
                }
            }
            else
                MessageBox.Show("An input value is invalid.");
        }

        // Ext Clock Value Check
        private void ExtClockValueCheck()
        {
            int tmpInt;
            if (ExtClockDisp.Text.Length >= 10)
                ExtClockDisp.Text = "";
            else if (int.TryParse(ExtClock.Text, out tmpInt))
                ExtClockDisp.Text = TUSBKDA.TUSBK01DA_ExtTimer(tmpInt).ToString("N0");
            else
                ExtClockDisp.Text = "";
        }

        private void ExtClock_TextChanged(object sender, EventArgs e)
        {
            ExtClockValueCheck();
        }

        // DAC Digital -> Analog Convert
        private void DacConvert_Click(object sender, EventArgs e)
        {
            short retErr;
            double Voltage;
            short iDigit;

            if( double.TryParse(DAC_DATA.Text,out Voltage) )
            {
                if( Voltage < -2.5 || Voltage > 2.5 )
                {
                    MessageBox.Show("入力値が不正です");
                    return;
                }
            }
            else
            {
                MessageBox.Show("入力値が不正です");
                return;
            }

            iDigit = (short)(Voltage / 2.5 * 8192); //Convert Voltage -> Integer
            if (iDigit < -8192)
                iDigit = -8192;
            else if (iDigit > 8191)
                iDigit = 8191;

            retErr = TUSBKDA.TUSBK01DA_DacOut(DevId, iDigit);
            if (retErr != 0)
                MessageBox.Show(TUSBKDA.GetErrMessage(retErr));

            if( DacStop.Enabled )
                EnableDisableCont(false);
        }

        private void IntClockValueCheck()
        {
            int tmpInt;

            if (IntClock.Text.Length < 11)
            {
                if (int.TryParse(IntClock.Text, out tmpInt))
                    IntClockDisp.Text = TUSBKDA.TUSBK01DA_IntClk(tmpInt).ToString("N0");
                else
                    IntClockDisp.Text = "";
            }
            else
                IntClockDisp.Text = "";
        }

        private void IntClock_TextChanged(object sender, EventArgs e)
        {
            IntClockValueCheck();
        }

        //矩形波生成
        private double RectangleF(double Arg)
        {
            Arg /= (2.0 * Math.PI);//スケール2Pi->1変換
            Arg -= Math.Floor(Arg);//鋸刃波形に変換
            if (Arg < 0.5)
                return 1.0;
            else
                return -1.0;
        }

        //三角波生成
        private double TriangleF(double Arg)
        {
            Arg /= (2.0 * Math.PI); //スケール2Pi->1変換
            Arg -= Math.Floor(Arg); //鋸刃波形に変換
            Arg *= 4.0; //スケール1 -> 4変換
            if( Arg < 1.0 )
                return Arg;
            else if( Arg < 3.0 )
                return (2.0 - Arg);
            else
                return (Arg - 4.0);
        }

        //波形生成
        private void MakeShape(ref double[] data, short Shape, int Leng, double Amp, double OFFSET)
        {
            int cnt;
            double angle;

            for (cnt = 0; cnt < Leng; cnt++)
            {
                angle = 2.0 * Math.PI * (double)cnt / (double)Leng;
                switch (Shape)
                {
                    case 0://正弦波
                        data[cnt] = Amp * System.Math.Sin(angle) + OFFSET;
                        break;
                    case 1://三角波
                        data[cnt] = Amp * TriangleF(angle) + OFFSET;
                        break;
                    case 2://矩形波
                        data[cnt] = Amp * RectangleF(angle) + OFFSET;
                        break;
                    default:
                        data[cnt] = 0;
                        break;
                }
            }
        }

        private void DacStart_Click(object sender, EventArgs e)
        {
            short prm_DataLeng;
            double prm_Ampl;
            double prm_Offset;
            int prm_IntClock;
            double[] DataD = new double[4095];
            short[] DataI = new short[4095];
            int cnt;

            //設定値チェック
            if (short.TryParse(DATA_LENG.Text, out prm_DataLeng) == false)
            {
                MessageBox.Show("DataLength設定に誤りがあります");
                return;
            }

            if (double.TryParse(AMPL.Text, out prm_Ampl ) == false)
            {
                MessageBox.Show("Amplitude設定に誤りがあります");
                return;
            }

            if (double.TryParse(OFFSET.Text, out prm_Offset) == false)
            {
                MessageBox.Show("Offset設定に誤りがあります");
                return;
            }

            if (SelIntClk.Checked)
            {
                if (int.TryParse(IntClock.Text, out prm_IntClock) == false)
                {
                    MessageBox.Show("IntClock設定に誤りがあります");
                    return;
                }
            }
            else
                prm_IntClock = 0;

            // 波形生成
            MakeShape(ref DataD, (short) WAVE_SHAPE.SelectedIndex, prm_DataLeng, prm_Ampl / 2.0, prm_Offset);

            //整数(DACコード)化
            for (cnt = 0; cnt < prm_DataLeng; cnt++)
            {
                DataI[cnt] = (short) ( DataD[cnt] * 8192.0 / 2.5 );
                if (DataI[cnt] >= 8191)
                    DataI[cnt] = 8191;
                else if (DataI[cnt] <= -8192)
                    DataI[cnt] = -8192;
            }

            //データ設定
            TUSBKDA.TUSBK01DA_Data_Set(DevId, DataI,  prm_DataLeng);

            //開始
            if (SelExtClock.Checked)
                TUSBKDA.TUSBK01DA_Da_Start(DevId, 0, prm_DataLeng);
            else
                TUSBKDA.TUSBK01DA_Da_Start(DevId, prm_IntClock, prm_DataLeng);

            EnableDisableCont(true);
        }

        private void DacStop_Click(object sender, EventArgs e)
        {
            short retErr;

            retErr = TUSBKDA.TUSBK01DA_Da_Stop(DevId);
            if (retErr != 0)
                MessageBox.Show(TUSBKDA.GetErrMessage(retErr));

            EnableDisableCont(false);
        }

        private void Timer1_Elapsed(object sender, System.Timers.ElapsedEventArgs e)
        {
            Timer1.Enabled = false;
            if (this.Visible == false)
                return;
            else
            {
                DIO_Check();
                Timer1.Enabled = true;
            }
        }
    }
}
