#//-----------------------------------------------
#//TUSB-16DIF用サンプルプログラム
#//Python 用モジュール
#//
#//2024_05_07　株式会社タートル工業
#//Copyright (C) 2024 Turtle Industry Co.,Ltd.
#//-----------------------------------------------
import TUSBDIF

from SpFuncs import *
from functools import partial

DevId = None                            # 開いているデバイスのID
mon_t = None                            # タイマー管理クラスインスタンス
lock = threading.Lock()                 # デバイスコマンドが同時実行されないために使用するロック
InputStat = [False for x in range(16)]  # 入力ピンの状態記録リスト

# Sub Functions -----------------------------

# Widgetの表示を設定する
# stat -> 'Close' , 'Open'
def WidgetState(stat):
    """Widgetの表示を設定する
    
    Arguments:
        stat        ----    設定状態 'Close' or 'Open'
    """
    EnClose = tk.NORMAL if( stat == 'Close' ) else tk.DISABLED
    EnOpen = tk.NORMAL if( stat == 'Open' ) else tk.DISABLED
    IDCmbb['state'] = EnClose
    OpenBtn['state'] = EnClose
    CloseBtn['state'] = EnOpen
    for bit in range(16):
        for sel in range(3):
            rbtn_seting[bit][sel]['state'] = EnOpen
    if stat != 'Close' :
        form.protocol('WM_DELETE_WINDOW',lambda: 'pass' )
    else :
        form.protocol('WM_DELETE_WINDOW',lambda: form.destroy())

def DirectionInit():
    """ピンを初期化、全て入力にする"""
    ret = TUSBDIF.Tusbdif_Low_Dir(DevId,0)      # Lowバイト全て入力に設定
    if ret != 0:
        ShowInfo(TUSBDIF.GetErrMessage(ret))
        return False
    ret = TUSBDIF.Tusbdif_High_Dir(DevId,0)     # Highバイト全て入力に設定
    if ret != 0:
        ShowInfo(TUSBDIF.GetErrMessage(DevId,0))
        return False
    for bit in range(16):
        SettingV[bit].set(0)
    return True

def DirectionSet():
    """ウィジットから入出力設定を読み込んで設定する"""
    PinDir = GetPinDir()
    ret = TUSBDIF.Tusbdif_Low_Dir(DevId,PinDir & 0xff) # Lowバイト入出力方向設定
    if ret != 0:
        ShowInfo(TUSBDIF.GetErrMessage(ret))
        return
    ret = TUSBDIF.Tusbdif_High_Dir(DevId,( PinDir >> 8 ) & 0xff) # Highバイト入出力方向設定
    if ret != 0:
        ShowInfo(TUSBDIF.GetErrMessage(ret))
        return 

def OutbitSet():
    """ウィジットから出力設定を読み込んで設定する"""
    ret = TUSBDIF.Tusbdif_Word_Write(DevId,GetPinOut()) # 出力ビット設定
    if ret != 0:
        ShowInfo(TUSBDIF.GetErrMessage(ret))
        return

def GetPinDir():
    """ピンの入出力方向指定値をウィジットから取得する"""
    DirBit = 0
    for bit in range(15,-1,-1):
        DirBit <<= 1
        tmp = SettingV[bit].get()
        if SettingV[bit].get() != 0:DirBit += 1
    return DirBit

def GetPinOut():
    """ピンの出力指定値をウィジットから取得する"""
    OutBit = 0
    for bit in range(15,-1,-1):
        OutBit <<= 1
        if SettingV[bit].get() ==2:OutBit += 1
    return OutBit

# Timer -------------------------------------
def Monitor():
    """入力状態モニタータイマー"""
    ret = TUSBDIF.Tusbdif_Word_Read(DevId)                  # 入力状態読み込み
    if ret[0] != 0: return
    for bit in range(16):
        if IsBitSet(ret[1],bit) != InputStat[bit]:
            InputStat[bit] = not InputStat[bit]
            SetLEDCanvas(led_pinstat[bit],InputStat[bit])

# Event Handler -----------------------------
def Setting_on(Bit):
    """ビット設定が変更された"""
    PinSet = SettingV[Bit].get()
    if PinSet == 0:                 # 入力に設定された場合
        DirectionSet()              # 入出力方向設定
    else:                           # 出力 Low or High　に設定された場合
        OutbitSet()                 # 出力ビット設定
        DirectionSet()              # 入出力方向設定

def OpenBtn_on():
    """オープンボタン押下"""
    global DevId
    DevId = IDCmbb.current()
    ret = TUSBDIF.Tusbdif_Device_Open(DevId)    # デバイスを開く
    if ret ==0:
        DirectionInit()
        WidgetState('Open')
        global mon_t
        mon_t = intTT(0.5,Monitor)
        mon_t.Start()
    else:
        ShowInfo(TUSBDIF.GetErrMessage(ret))

def CloseBtn_on():
    """クローズボタン押下"""
    mon_t.Stop()
    TUSBDIF.Tusbdif_Device_Close(DevId)         # デバイスを閉じる
    WidgetState('Close')

# form generation ---------------------------
form  = tk.Tk()
form.geometry('400x500')
form.title('TUSB-16DIF Sample Program')
form.attributes("-toolwindow",True)

# 見出しラベル
GetLabel(form,"Bit",0.05,0.05)
GetLabel(form,"Setting",0.15,0.05)
GetLabel(form,"PinStatus",0.75,0.05)

# ビット毎の設定、表示
PinSettingDic= {
    "INPUT":0,
    "OUT(L)":1,
    "OUT(H)":2,
}
rbtn_seting = GenList2D(16,3,None)      # ピン選択ラジオボタン
SettingV = GenTkVarList(16,tk.IntVar,0) # ピン選択ラジオボタンの設定値
led_pinstat = GenList(16,None)          # LEDタイプ表示

bypit = 0.046    # 行ピッチ
for bit in range(16):
    ly = 0.1+bypit*bit
    GetLabel(form,str(bit),0.05,ly)
    rbtn_seting[bit] = GetRadioButtons(form,PinSettingDic,0.15,ly,0.15,0,SettingV[bit],partial(Setting_on,Bit=bit))
    led_pinstat[bit] = GetLEDCanvas(form,0.80,ly)
    SetLEDCanvas(led_pinstat[bit],False)

#ID open close ###################
IDList = [str(i) for i in range(16)]
GetLabel(form,"ID",0.05,0.9)
IDCmbb = GetComboBox(form,IDList,0.15,0.15,0.9)
OpenBtn = GetButton(form,"DevOpen",OpenBtn_on,0.2,0.5,0.9)
CloseBtn = GetButton(form,"DevClose",CloseBtn_on,0.2,0.8,0.9)

# Start
WidgetState('Close')
form.mainloop()
