#//-----------------------------------------------
#//TUSB-16DIF用ドライバアクセス用API
#//Python 用モジュール
#//
#//2024_05_02　株式会社タートル工業
#//Copyright (C) 2024 Turtle Industry Co.,Ltd.
#//-----------------------------------------------

from ctypes import *

tusbdifdll = cdll.LoadLibrary("TUSBDIF.DLL")

tusbdifdll.Tusbdif_Device_Open.restype = c_short
tusbdifdll.Tusbdif_Device_Open.argtypes = (c_short,)
tusbdifdll.Tusbdif_Device_Close.restype = None
tusbdifdll.Tusbdif_Device_Close.argtypes = (c_short,)
tusbdifdll.Tusbdif_Low_Dir.restype = c_short
tusbdifdll.Tusbdif_Low_Dir.argtypes = (c_short,c_ubyte)
tusbdifdll.Tusbdif_High_Dir.restype = c_short
tusbdifdll.Tusbdif_High_Dir.argtypes = (c_short,c_ubyte)
tusbdifdll.Tusbdif_LowByte_Write.restype = c_short
tusbdifdll.Tusbdif_LowByte_Write.argtypes = (c_short,c_ubyte)
tusbdifdll.Tusbdif_HighByte_Write.restype = c_short
tusbdifdll.Tusbdif_HighByte_Write.argtypes = (c_short,c_ubyte)
tusbdifdll.Tusbdif_LowByte_Read.restype = c_short
tusbdifdll.Tusbdif_LowByte_Read.argtypes = (c_short,POINTER(c_ubyte))
tusbdifdll.Tusbdif_HighByte_Read.restype = c_short
tusbdifdll.Tusbdif_HighByte_Read.argtypes = (c_short,POINTER(c_ubyte))
tusbdifdll.Tusbdif_Word_Write.restype = c_short
tusbdifdll.Tusbdif_Word_Write.argtypes = (c_short,c_uint)
tusbdifdll.Tusbdif_Word_Read.restype = c_short
tusbdifdll.Tusbdif_Word_Read.argtypes = (c_short,POINTER(c_uint))

#######  support functions (don not call directry from user codes)

def isint(n):
    return type(n) is int

def IsInRange(Val,Min,Max):
    if(isint(Val) == False ):   return False
    elif(Val < Min):            return False
    elif(Val > Max):            return False
    else:                       return True

def IsValidId(DevId):
    return IsInRange(DevId,0,15)

# def functions ------------------------------------

def Tusbdif_Device_Open(DevId):
    """Open Device Connection
    
    Arguments:
    DevId           ----    DevIce ID 0~15
    
    Returns:
    int             ----    Result code
    """
    if(not IsValidId(DevId)):return 1
    return tusbdifdll.Tusbdif_Device_Open(DevId)

def Tusbdif_Device_Close(DevId):
    """Close Device Connection

    Arguments:
    DevId           ----    Device ID 0~15

    Returns:
    None
    """
    if(not IsValidId(DevId)):return 2
    tusbdifdll.Tusbdif_Device_Close(DevId)

def Tusbdif_Low_Dir(DevId,Data):
    """Set Low Byte Direction
    
    Arguments:
    DevId           ----    Device ID 0~15
    Deta            ----    Output Direction bit pattern 0~255

    Returns:
    int             ----    Result code
    """
    if(not IsValidId(DevId)):return 2
    if(not IsInRange(Data,0,255)):return 2
    return tusbdifdll.Tusbdif_Low_Dir(DevId,Data)

def Tusbdif_High_Dir(DevId,Data):
    """Set High Byte Direction
    
    Arguments:
    DevId           ----    Device ID 0~15
    Deta            ----    Output Direction bit pattern 0~255

    Returns:
    int             ----    Result code
    """
    if(not IsValidId(DevId)):return 2
    if(not IsInRange(Data,0,255)):return 2
    return tusbdifdll.Tusbdif_High_Dir(DevId,Data)

def Tusbdif_LowByte_Write(DevId,Data):
    """Set Low Byte Output
    
    Arguments:
    DevId           ----    Device ID 0~15
    Deta            ----    Output bit pattern 0~255

    Returns:
    int             ----    Result code
    """
    if(not IsValidId(DevId)):return 2
    if(not IsInRange(Data,0,255)):return 2
    return tusbdifdll.Tusbdif_LowByte_Write(DevId,Data)

def Tusbdif_HighByte_Write(DevId,Data):
    """Set High Byte Output
    
    Arguments:
    DevId           ----    Device ID 0~15
    Deta            ----    Output bit pattern 0~255

    Returns:
    int             ----    Result code
    """
    if(not IsValidId(DevId)):return 2
    if(not IsInRange(Data,0,255)):return 2
    return tusbdifdll.Tusbdif_HighByte_Write(DevId,Data)

def Tusbdif_LowByte_Read(DevId):
    """Read Low Byte Input
    
    Arguments:
    DevId           ----    Device ID 0~15
    
    Returns:
    int,            ----    Result code
    int,            ----    Input Data
    """
    if(not IsValidId(DevId)):return 2
    Data = c_ubyte()
    ret = tusbdifdll.Tusbdif_LowByte_Read(DevId,byref(Data))
    if ret == 0:
        return 0,Data.value
    else:
        return ret,None
    
def Tusbdif_HighByte_Read(DevId):
    """Read High Byte Input
    
    Arguments:
    DevId           ----    Device ID 0~15
    
    Returns:
    int,            ----    Result code
    int,            ----    Input Data
    """
    if(not IsValidId(DevId)):return 2
    Data = c_ubyte()
    ret = tusbdifdll.Tusbdif_HighByte_Read(DevId,byref(Data))
    if ret == 0:
        return 0,Data.value
    else:
        return ret,None

def Tusbdif_Word_Write(DevId,Data):
    """Set Output to 16bit width
    
    Arguments:
    DevId           ----    Device ID 0~15
    Deta            ----    Output bit pattern 0~65535

    Returns:
    int             ----    Result code
    """
    if(not IsValidId(DevId)):return 2
    if(not IsInRange(Data,0,65535)):return 2
    return tusbdifdll.Tusbdif_Word_Write(DevId,Data)

def Tusbdif_Word_Read(DevId):
    """Read Input in 16bit width
    
    Arguments:
    DevId           ----    Device ID 0~15
    
    Returns:
    int,            ----    Result code
    int,            ----    Input Data
    """
    if(not IsValidId(DevId)):return 2
    Data = c_uint()
    ret = tusbdifdll.Tusbdif_Word_Read(DevId,byref(Data))
    if ret == 0:
        return 0,Data.value
    else:
        return ret,None

#戻り値用エラーメッセージ
ErrStrDic = {
            0:"正常終了しました",
            1:"ID番号が不正です",
            2:"ドライバがインストールされていません",
            3:"すでにデバイスはオープンされています",
            4:"接続されている台数が多すぎます",
            5:"オープンできませんでした",
            6:"デバイスがみつかりません",
            8:"パラメータエラーです",
            9:"USB通信エラーです"
}
def GetErrMessage(retcode):
    if( retcode in ErrStrDic):
        return ErrStrDic[retcode]
    else:
        return "不明なエラーです"

